
// Example program that uses the LCD4Bit library
// with the ZB1-LCD2 board.

// The ZB1-LCD2 has two pusbuttons (S11,S12), 
// two LEDs (D2,D4) and a buzzer (SPKR1).
// There is also a bit to turn the backlight LED on and off.


#include <avr/interrupt.h>    
#include <avr/io.h>  
#include <LCD4Bit.h> 

#define OFF LOW
#define ON  HIGH

// Hardware on the ZB1 board
                         
#define STATUS_LED      7 // ZB1    (PD7)

// Hardware on the ZB1-LCD2

#define LED1            3  // D4    (PD3)
#define LED2            4  // D2    (PD4)  
#define BUZZER         10  // SPKR1 (PB2)
#define LCD_BACKLIGHT   5  // Q1    (PD5)
#define BUTTON1       PC4  // S11   (PC4)
#define BUTTON2       PC5  // S12   (PC5)

// button interrupts

#define BUTTON1_INT PCINT12
#define BUTTON2_INT PCINT13

boolean volatile button1 = LOW;
boolean volatile button2 = LOW;

#define set_backlight(state)  digitalWrite(LCD_BACKLIGHT, state ? HIGH : LOW)
#define set_buzzer(state)     digitalWrite(BUZZER, state)
#define set_led1(state)       digitalWrite(LED1, state)
#define set_led2(state)       digitalWrite(LED2, state)

LCD4Bit lcd = LCD4Bit(2); 

void setup() {
  char *hello = "Hello";
  char *world = "World";
  pinMode(LED1, OUTPUT);
  pinMode(LED2, OUTPUT);
  pinMode(STATUS_LED, OUTPUT);  //we'll use the debug LED to output a heartbeat
  pinMode(LCD_BACKLIGHT, OUTPUT); // backlight
  pinMode(BUZZER, OUTPUT);
  set_buzzer(LOW);
  set_led1(LOW);
  set_led2(LOW);
  set_backlight(OFF);
  lcd.init();
  lcd.clear();
  lcd.printIn(hello);
  lcd.cursorTo(2,0);
  lcd.printIn(world);
  blink(3);
  //
  // PCI1 will trigger if any enabled PCINT14..8 pin toggles
  // 
  PCICR  |= (1 << PCIE1);
  PCMSK1 |= (1 << BUTTON1_INT);
  PCMSK1 |= (1 << BUTTON2_INT);  
}

void loop() {
  if (button1) set_led1(ON);
  else set_led1(OFF);
  if (button2) {
    set_led2(ON);
    beep(200);
    button2 = 0;
    set_led2(OFF);
  }
}

void beep(int n) {
 for  (int i = 0; i < n; i++) {
    digitalWrite(BUZZER, HIGH);
    delay(1);
    digitalWrite(BUZZER, LOW);
    delay(1);  
  }
}

// Blink the status LED

void blink(int n) {
  for (int i = 0; i < n; i++) {
    digitalWrite(STATUS_LED, HIGH);
    delay(200);
    digitalWrite(STATUS_LED, LOW);
    delay(200);  
  }
  delay(1000);
}
 
// button1 toggles the backlight
// button2 beeps the piezo speaker

ISR( PCINT1_vect )
{
  if (PINC & _BV(BUTTON1)) {
    if (button1) {
      button1 = 0;
      set_backlight(OFF);
    } else {
      button1 = 1;
      set_backlight(ON);
    }
  }
  if (PINC & _BV(BUTTON2)) {
    if (button2 == 0) button2 = 1;
  }
}
 
