///
/// @file
///
#include <NCO.h>
#include <inttypes.h>
#include <avr/interrupt.h>
#include <avr/io.h>
#include "WConstants.h"

struct nco_struct nco[NCO_NUM_CHS-1];

/// \brief Call this function to increment all of the phase accumulators
///
/// \return VOID
///
void nco_inc(void) {
  unsigned char i;
  for (i = 0; i < NCO_NUM_CHS; i++) nco_inc_phase_acc(i);
}

/// \brief Sets the phase increment of an NCO channel
///
/// The value of the frequency must be less than half the sampling
/// rate.
/// 
/// \param[in] ch oscillator channel number (0..3)
/// \param[in] freq oscillator frequency
///
/// \return VOID
///
void nco_set_freq(unsigned char ch, double freq) {
  nco[ch].inc = (unsigned long)(freq * NCO_NUM_PTS * (unsigned long)(NCO_SCALE / NCO_SAMPLING_RATE));
}

///
/// \brief Initializes the NCO structure
///
/// Resets the phase accumulator, phase increment and the DAC control
/// bytes. This should be called once at the start of program before
/// calling any other NCO functions or enabling interrupts.
///
void nco_init(void) {
  unsigned char i;
  for (i = 0; i < NCO_NUM_CHS; i++) {
    nco[i].acc.acc_long = 0;
    nco[i].inc = 0;
    switch(i) {
    case 0: nco[i].control = NCO_CH0_CONTROL; break;
    case 1: nco[i].control = NCO_CH1_CONTROL; break;
    case 2: nco[i].control = NCO_CH2_CONTROL; break;
    case 3: nco[i].control = NCO_CH3_CONTROL; break;
    }
  }
  PORTB |= (1<<NCO_DAC_LOAD) | (1<<NCO_DAC_LATCH);
}


///
/// \brief toggles the NB1A DAC latch bit low then high
///
#define nco_dac_toggle_latch { PORTB&=~(1<<NCO_DAC_LATCH); PORTB|=(1<<NCO_DAC_LATCH);}

///
/// \brief toggles the NB1A DAC load bit low then high
///
#define nco_dac_toggle_load  { PORTB&=~(1<<NCO_DAC_LOAD); PORTB|=(1<<NCO_DAC_LOAD);}


///
/// \brief Updates all of the NCO channels
/// 
/// This function should be called every time the timer interrupt occurs.
/// 
void nco_update(void) {
  unsigned char i;
  unsigned char addr;
  cli();
  for (i = 0; i < NCO_NUM_CHS; i++) {
    nco_inc_phase_acc(i);
    SPDR = nco[i].control;
    addr = nco_addr(i);
    if (nco_round(i)) {
      addr += 1;
      addr &= NCO_ADDR_MASK;
    }
    while((SPSR & (1<<SPIF)) == 0);
    SPDR = nco_pts[addr];
    while((SPSR & (1<<SPIF)) == 0);
    nco_dac_toggle_load;
    nco_dac_toggle_latch;
  }
  sei();
}

