/// \file
///
/// Numerically controlled oscillator
///
#ifndef __NCO_H__
#define __NCO_H__

#include <inttypes.h>

///
/// NCO Wavetable (generated by the Perl script nco-data)
///

#define NCO_ADDR_LEN 6
#define NCO_NUM_PTS 64
#define NCO_LAST_PT 63
static unsigned char nco_pts[] = {
    128, 140, 152, 165, 176, 188, 198, 208,
    218, 226, 234, 240, 245, 250, 253, 254,
    255, 254, 253, 250, 245, 240, 234, 226,
    218, 208, 198, 188, 176, 165, 152, 140,
    128, 115, 103,  90,  79,  67,  57,  47,
     37,  29,  21,  15,  10,   5,   2,   1,
      0,   1,   2,   5,  10,  15,  21,  29,
     37,  47,  57,  67,  79,  90, 103, 115};

///
/// One TLV5620 channel for each NCO channel. 
///

#define NCO_NUM_CHS 4

///
/// Six bits are used in the address calculation (64 byte wavetable). 
/// Change this value if the size of the wavetable is changed.
///
#define NCO_ADDR_MASK 0x3F

///
/// Control bytes for the TLV5620 DAC
///

#define NCO_CH0_CONTROL ((0<<1) | (0x01))
#define NCO_CH1_CONTROL ((1<<1) | (0x01))
#define NCO_CH2_CONTROL ((2<<1) | (0x01))
#define NCO_CH3_CONTROL ((3<<1) | (0x01))

///
/// Prescaler Constants
///

#define NCO_CLOCK_NONE    ((0<<CS22) | (0<<CS21) | (0<<CS20))
#define NCO_CLOCK_DIV1    ((0<<CS22) | (0<<CS21) | (1<<CS20))
#define NCO_CLOCK_DIV8    ((0<<CS22) | (1<<CS21) | (0<<CS20))
#define NCO_CLOCK_DIV32   ((0<<CS22) | (1<<CS21) | (1<<CS20))
#define NCO_CLOCK_DIV64   ((1<<CS22) | (0<<CS21) | (0<<CS20))
#define NCO_CLOCK_DIV128  ((1<<CS22) | (0<<CS21) | (1<<CS20))
#define NCO_CLOCK_DIV256  ((1<<CS22) | (1<<CS21) | (0<<CS20))
#define NCO_CLOCK_DIV1024 ((1<<CS22) | (1<<CS21) | (1<<CS20))

///
/// The TLV5620 load and latch pins are connected to
/// Port B pins 2 and 1 respectively.
///

#define NCO_DAC_LOAD  PB2
#define NCO_DAC_LATCH PB1

///
/// NCO wavetable is 64 entries. The sample rate is 3750Hz.  For the
/// NB1A the fclk_io = 12MHz For a sample rate of 3750 the prescaler
/// is set to of eight and the count register (OCR2A) is set to 99
///

#define NCO_SAMPLING_RATE 3750 // Hz
#define NCO_NUM_UNUSED_BITS 32 - NCO_ADDR_LEN - 1
#define NCO_SCALE 0x04000000
#define NCO_OCR2A 99

///
/// NCO Structure
///
/// Contains the phase accumulator and phase increment for the numerically
/// controlled oscillator. Both of these values are longs but only the
/// address and round bits are used to access the wavetable.
/// 
/// avr-gcc (4.1.2) produced an error with bit fields larger than
/// 16 bits. The number of unused bits for most NCO applications will
/// probably exceed 16 bits so two bit fields are declared.
///

struct nco_struct {
  union {
    struct {
      unsigned UNUSED1 : 16;
      unsigned UNUSED2 : NCO_NUM_UNUSED_BITS - 16;
      unsigned round   : 1;
      unsigned address : NCO_ADDR_LEN;
    } acc_bits;
    unsigned long acc_long;
  } acc;                 // phase accumulator
  unsigned long inc;     // phase increment
  unsigned char control; // control byte for the TLV5620
};
extern struct nco_struct nco[NCO_NUM_CHS-1];

///
/// Macros for initializing each channel of the NCO and incrementing
/// the phase accumulator.
///

#define nco_init_ch(i,control) { nco[i].acc.acc_long = 0; nco[i].inc = 0; nco[i].control = control;}
#define nco_inc_phase_acc(i) { nco[i].acc.acc_long += nco[i].inc; }

///
/// Macros for accessing a wavetable point, the point address and the
/// rounding bit.  
///

#define nco_addr(i)  (nco[i].acc.acc_bits.address)
#define nco_round(i) (nco[i].acc.acc_bits.round)
#define nco_point(i) (nco_pts[ nco_addr(i) ])

///
/// Predeclaration of procedures
///

void nco_set_freq(unsigned char ch, double freq);
void nco_inc(void);
void nco_init(void);
void nco_update(void);

#endif // __NCO_H__
