// RTC clock library for the NB1A

// DS1337 IC


#include "RTC.h"

extern "C" 
{
#include <inttypes.h>
#include "WConstants.h"
#include "TWI.h"
}


/// \class RTC
/// \brief Class for the Maxim DS1337 Real-time Clock IC.
/// 
/// The main data structure in the RTC class enables access to the
/// DS1337 register values as bit fields, bytes, or an array of
/// bytes. The union <code>_regs</code> creates overlays three data
/// structures -- <code>reg_bits</code> (bit fields),
/// <code>reg_bytes</code> (byte fields) and <code>reg_array</code>
/// (array of bytes).  <br>
/// 

///
/// Constructs an RTC object. 
/// \todo Add intialization for the interrupt line connections.
/// By default alarm output 1 is connected to Pin 6 (PD6) and
/// alarm output 2 is connected to Pin 5 (PD5)
///

RTC::RTC() {
  
};

#define _rtc_reg_p_(reg) (reg <= RTC_LAST_REG ? 1 : 0)

///
/// Writes a value to a DS1337 register
/// \param[in] reg DS1337 register
/// \param[in] v value to write
/// \retval  0 success
/// \retval -1 invalid parameter value
/// \todo Add TWI transmit error codes
///
char RTC::write_reg(unsigned char reg, unsigned char v) {
  if (!_rtc_reg_p_(reg)) return(-1);
  _buf[0] = _i2c_wr_address;
  _buf[1] = reg;
  _buf[2] = v;
  twi_transmit(_buf, 3);
  return(0);
}

///
/// Writes values to a sequential block of registers.
/// \param[in] reg first register in the block
/// \param[in] *v pointer to the block of values to write
/// \param[in] n number of values in the block
/// \retval  0 success
/// \retval -1 invalid parameter value
/// \todo Add TWI transmit error codes
///
char RTC::write_regs(unsigned char reg, unsigned char *v, unsigned char n) {
  unsigned char *p;
  unsigned char i;
  if (!_rtc_reg_p_(reg)) return(-1);
  if (n > 16) return(-1);
  _buf[0] = _i2c_wr_address;
  _buf[1] = reg;
  p = v;
  for (i = 0; i < n; i++) _buf[i + 2] = *p++;
  twi_transmit(_buf, n + 2);
  return(0);
}

///
/// Sets the DS1337 register pointer
/// \param[in] reg register number
/// \retval 0 success
/// \retval -1 invalid parameter value
/// \todo Add error codes
///

char RTC::write_reg_ptr(unsigned char reg) {
  if (!_rtc_reg_p_(reg)) return(-1);
  _buf[0] = _i2c_wr_address;
  _buf[1] = reg;
  twi_transmit(_buf, 2);
  return(0);
}

///
/// Reads all of the DS1337 registers into the RTC data structure
/// \retval  0 success
/// \retval -1 TWI transmit error
///

char RTC::read_regs() {
  unsigned char i;
  write_reg_ptr(0);
  _buf[0] = _i2c_rd_address;
  twi_transmit(_buf, 17); // 16 register bytes + ADDR
  delay(10);
  while(twi_busy_p());
  // retrieve the data bytes from the twi buffer
  if (!twi_status_reg.last_trans_ok) return(-1);
  twi_get_data(_regs.reg_array, 17);
  return(0);
}


///
/// Reads and returns the value of a DS1337 register.
/// \param[in] reg DS1337 register
/// \returns the value for DS1337 register <code>reg</code>
/// \todo Add TWI transmit error codes
///

unsigned char RTC::read_reg(unsigned char reg) {
  unsigned char i;
  unsigned char data[1];
  if (!_rtc_reg_p_(reg)) return(-2);
  write_reg_ptr(reg);
  _buf[0] = _i2c_rd_address;
  _buf[1] = reg;
  delay(10);
  twi_transmit(_buf, 2); // addr + reg
  while(twi_busy_p());
  // retrieve the data bytes from the twi buffer
  if (!twi_status_reg.last_trans_ok) return(-1);
  twi_get_data(data, 2);
  return(data[1]);
}


#define PM_BIT 0x02
#define _rb_(name) (_regs.reg_bits.name)   ///< register bits for <B>name</B>
#define _digit_(name) (_rb_(name) + '0')   ///< ASCII digit
#define _hours_10_ ( _rb_(hour12) ? (_rb_(hours_10) & 0x01) : _rb_(hours_10)) ///< left digit of hours
#define _hours_    ( _rb_(hours) )
#define _pm_p_ (_rb_(hours_10) & PM_BIT)

///
/// Returns the last value of seconds register that was read
/// 
unsigned char RTC::get_secs()  { return(_rb_(secs_10) * 10 + _rb_(secs)); }
///
/// Returns the last value of minutes register that was read
///
unsigned char RTC::get_mins()  { return(_rb_(mins_10) * 10 + _rb_(mins)); }
///
/// Returns the last value of hours register that was read
///
unsigned char RTC::get_hours() { return(_hours_10_ * 10 + _hours_); }
///
/// Returns the last value of the day register that was read
///
unsigned char RTC::get_day()   { return(_rb_(day_10) * 10 + _rb_(day)); }
///
/// Returns the last value of the month register that was read
///
unsigned char RTC::get_month() { return(_rb_(month_10) * 10 + _rb_(month)); }
///
/// Returns the last value of the year register that was read
///
unsigned int  RTC::get_year()  { return(2000 + _rb_(year_10) * 10 + _rb_(year)); }

// breaks a binary value into two BCD digits

#define _break_(x, x10, x1) { x1  = x % 10; x10 = (unsigned char)((x - x1) / 10); }


///
/// Macros to perform range checking on year, month, day 
///
#define valid_year_p(y)  (((y > 2099) || (y < 2000)) ? 0 : 1)
#define valid_month_p(m) (((m > 12)   || (m < 1))    ? 0 : 1)
#define valid_day_p(d)   (((d > 31)   || (d < 1))    ? 0 : 1)


///
/// Sets the RTC year register.
/// \param[in] year valid values are from 2000..2099 inclusive
/// \retval  0 success
/// \retval -1 invalid parameter value
///
unsigned char RTC::set_year(unsigned int year) {
  unsigned char y10;
  unsigned char y1; 
  if (year > 2099) return(-1);
  if (year < 2000) return(-1);
  year -= 2000;
  _break_(year, y10, y1);
  write_reg(RTC_REG_YEAR, (y10 << 4) | y1);
  return(0);
}

///
/// Sets the RTC month register.
/// \param[in] month valid values are from 1..12 inclusive
/// \retval  0 success
/// \retval -1 invalid parameter value
///
unsigned char RTC::set_month(unsigned char month) {
  unsigned char m10, m1;
  if (month > 12) return(-1);
  if (month < 1)  return(-1);
  _break_(month, m10, m1);
  write_reg(RTC_REG_MONTH, (m10 << 4) | m1);
  return(0);
}

///
/// Sets the RTC day register.
/// \param[in] day valid values are from 1..31 inclusive
/// \retval  0 success
/// \retval -1 invalid parameter value
///
unsigned char RTC::set_day(unsigned char day) {
  unsigned char d10, d1;
  if (day > 31) return(-1);
  if (day < 1)  return(-1);
  _break_(day, d10, d1);
  write_reg(RTC_REG_DAY, (d10 << 4) | d1);
  return(0);
}

///
/// Sets the RTC date.
/// \param[in] year valid values are from 2000..2099 inclusive
/// \param[in] month valid values are from 1..12 inclusive
/// \param[in] day valid values are from 1..31 inclusive
/// \retval  0 success
/// \retval -1 invalid parameter value
/// \todo check for valid values for year, month and day prior to calling any functions
///
unsigned char RTC::set_date(unsigned int year, unsigned char month, 
			    unsigned char day) {
  if (set_year(year)) return(-1);
  if (set_month(month)) return(-1);
  if (set_day(day)) return(-1);
  return(0);
}


///
/// Sets the RTC seconds register.
/// \param[in] secs valid values are from 0..59 inclusive
/// \retval  0 success
/// \retval -1 invalid parameter value
///
unsigned char RTC::set_secs(unsigned char secs) {
  unsigned char s10, s1;
  if (secs > 59) return(-1);
  _break_(secs, s10, s1);
  write_reg(RTC_REG_SECS, (s10 << 4) | s1);
  return(0);
}

///
/// Sets the RTC minutes register.
/// \param[in] mins valid values are from 0..59 inclusive
/// \retval  0 success
/// \retval -1 invalid parameter value
///
unsigned char RTC::set_mins(unsigned char mins) {
  unsigned char m10, m1;
  if (mins > 59) return(-1);
  _break_(mins, m10, m1);
  write_reg(RTC_REG_MINS, (m10 << 4) | m1);
  return(0);
}

/// \cond test
union hours_reg {
  struct {
    unsigned hours    : 4;
    unsigned hours_10 : 1;
    unsigned pm       : 1;
    unsigned hour12   : 1;
    unsigned UNUSED   : 1;
  } _bits12; ///< 12 hour mode bit fields
  struct {
    unsigned hours    : 4;
    unsigned hours_10 : 2;
    unsigned hour12   : 1;
    unsigned UNUSED   : 1;
  } _bits24; ///< 24 hour mode bit fields
  unsigned char _byte; ///< hours data byte
};

/// \endcond

///
/// \union hours_reg
///

///
/// Sets the RTC hours register, in 12 hour mode, to <B>hours</B>. If <B>hours</B>
/// is greater than 12 than an error code is returned and the RTC register is not
/// changed.
///

unsigned char RTC::set_hours12(unsigned char hours, unsigned char pm_p) {
  union hours_reg reg;
  if (hours > 12) return(-1);
  reg._bits12.hour12 = 1;
  reg._bits12.pm = pm_p ? 1 : 0;
  _break_(hours, reg._bits12.hours_10, reg._bits12.hours);
  write_reg(RTC_REG_HOURS, reg._byte);
  return(0);
}

///
/// Sets the RTC hours register, in 24 hour mode, to <B>hours</B>. If <B>hours</B>
/// is greater than 23 than an error code is returned and the RTC register is not
/// changed.
///

unsigned char RTC::set_hours(unsigned char hours) {
  union hours_reg reg;
  if (hours > 23) return(-1);
  reg._bits24.hour12 = 0;
  _break_(hours, reg._bits24.hours_10, reg._bits24.hours);
  write_reg(RTC_REG_HOURS, reg._byte);
  return(0);
}

///
/// Sets the RTC time (24 hour mode) to <B>hours</B>, <B>mins</B>, <B>secs</B>
///

unsigned char RTC::set_time(unsigned char hours, unsigned char mins, 
			    unsigned char secs) {
  if (set_hours(hours)) return(-1);
  if (set_mins(mins)) return(-1);
  if (set_secs(secs)) return(-1);
  return(0);
}

///
/// Sets the RTC time (12 hour mode).
/// \param[in] hours
/// \param[in] mins
/// \param[in] secs
/// \param[in] pm_p =1 for PM, =0 for AM
///

unsigned char RTC::set_time12(unsigned char hours, 
			      unsigned char mins, 
			      unsigned char secs,
			      unsigned char pm_p) {
  if (set_hours12(hours, pm_p)) return(-1);
  if (set_mins(mins)) return(-1);
  if (set_secs(secs)) return(-1);
  return(0);
}

#define _ar_(n) alarm_reg._bits.n
#define _a1_bits_(dy, m4, m3, m2, m1) { _ar_(a1_dy) = dy; \
                                        _ar_(a1m4) = m4; _ar_(a1m3) = m3; \
                                        _ar_(a1m2) = m2; _ar_(a1m1) = m1; }


///
/// Setup alarm mode and conditions for alarm1. After the alarm is setup
/// it needs to be enabled using <code>enable_alarm1</code>.
///
/// \param[in] mode one of values listed below. 
/// <dl>
/// <dt>RTC_ALARM1_MODE1</dt>&nbsp;&nbsp; once per second
/// <dt>RTC_ALARM1_MODE2</dt>&nbsp;&nbsp; when the seconds match
/// <dt>RTC_ALARM1_MODE3</dt>&nbsp;&nbsp; when the minutes and seconds match
/// <dt>RTC_ALARM1_MODE4</dt>&nbsp;&nbsp; when the hours, minutes, seconds, match
/// <dt>RTC_ALARM1_MODE5</dt>&nbsp;&nbsp; when the day, hours, minutes, seconds, match
/// <dt>RTC_ALARM1_MODE6</dt>&nbsp;&nbsp; when the dow, hours, minutes, seconds, match
/// </dl>
/// <p>
/// <p>
/// \param[in] day_or_dow 1..31 for <code>RTC_ALARM_MODE5</code> 1..7 for <code>RTC_ALARM_MODE6</code>
/// \param[in] hours 0..23
/// \param[in] mins 0..59
/// \param[in] secs 0..59
/// \todo add return error codes
/// \todo add bounds checking for parameters
///
unsigned char RTC::set_alarm1(unsigned char mode, unsigned char day_or_dow,
			      unsigned char hours, unsigned char mins, 
			      unsigned char secs) {
  union {
    struct {
      unsigned a1_secs       : 4;
      unsigned a1_secs_10    : 3;
      unsigned a1m1          : 1;
      
      unsigned a1_mins       : 4;
      unsigned a1_mins_10    : 3;
      unsigned a1m2          : 1;

      unsigned a1_hours      : 4;
      unsigned a1_hours_10   : 2;
      unsigned a1_hour12     : 1;
      unsigned a1m3          : 1;

      unsigned a1_day        : 4;
      unsigned a1_day_10     : 2;
      unsigned a1_dy         : 1;
      unsigned a1m4          : 1;
    } _bits;
    unsigned char _bytes[3];
  } alarm_reg = { 0, 0, 0, 0 };
  switch(mode) {
  case RTC_ALARM1_MODE1: _a1_bits_(0, 1, 1, 1, 1); break;
  case RTC_ALARM1_MODE2: _a1_bits_(0, 1, 1, 1, 0); break;
  case RTC_ALARM1_MODE3: _a1_bits_(0, 1, 1, 0, 0); break;
  case RTC_ALARM1_MODE4: _a1_bits_(0, 1, 0, 0, 0); break;
  case RTC_ALARM1_MODE5: _a1_bits_(0, 0, 0, 0, 0); break;
  case RTC_ALARM1_MODE6: _a1_bits_(1, 0, 0, 0, 0); break;
  default:               _a1_bits_(0, 0, 0, 0, 0); break;
  }
  _break_(secs,       alarm_reg._bits.a1_secs_10,  alarm_reg._bits.a1_secs);
  _break_(mins,       alarm_reg._bits.a1_mins_10,  alarm_reg._bits.a1_mins);
  _break_(hours,      alarm_reg._bits.a1_hours_10, alarm_reg._bits.a1_hours);
  _break_(day_or_dow, alarm_reg._bits.a1_day_10,   alarm_reg._bits.a1_day);

  write_regs(RTC_REG_A1SECS, alarm_reg._bytes, 4);

  return(0);
}

///
/// Clears the alarm1 interrupt flag.
/// \todo add return error codes
///
unsigned char RTC::clear_alarm1(void) {
  unsigned char reg_value;
  reg_value = read_reg(RTC_REG_STATUS);
  reg_value &= ~RTC_A1F;
  write_reg(RTC_REG_STATUS, reg_value);
  return(0);
}

///
/// Enables the interrupt for alarm1
/// \todo add return error codes
///
unsigned char RTC::enable_alarm1(void) {
  unsigned char reg_value;
  clear_alarm1();
  reg_value = read_reg(RTC_REG_CONTROL);
  reg_value |= RTC_A1IE;
  write_reg(RTC_REG_CONTROL, reg_value);
  return(0);
}


#define _substr_(str) { *p++ = _digit_(str##_10); *p++ = _digit_(str); }

///
/// Returns a 23 character null terminated string containing the most
/// recent date and time that was read from the DS1337.  The string
/// format is yyyy-mm-dd hh:mm:mm AM 
///
/// \param[out] *str pointer to memory location that will contain the
/// output string
///

void RTC::localtime(char *str) {
  char *p;
  p = str;
  *p++ = '2';
  *p++ = '0';
  _substr_(year);  *p++ = '-';
  _substr_(month); *p++ = '-';
  _substr_(day);   *p++ = ' ';
  *p++ = _hours_10_ + '0';
  *p++ = _hours_ + '0';
  *p++ = ':';
  _substr_(mins);  *p++ = ':';
  _substr_(secs);  
  if (_rb_(hour12)) {
    *p++ = ' ';
    *p++ = _pm_p_ ? 'P' : 'A';
    *p++ = 'M';
  }
  *p   = '\0';
}

///
/// Returns the last read value of RTC register <code>reg</code>
/// \param[in] reg DS1337 register number.
/// \returns the last read value of <code>reg</code> from the DS1337 data structure.
///
unsigned char RTC::get_reg(unsigned char reg) {
  if (reg > RTC_LAST_REG) return(0xFF);
  return(_regs.reg_array[reg + 1]);
}

