#ifndef __RTC_H__
#define __RTC_H__

#include <inttypes.h>

///
/// Time of day registers.
/// See the DS1337 datasheet page 8
///

#define RTC_REG_SECS      0x00
#define RTC_REG_MINS      0x01
#define RTC_REG_HOURS     0x02
#define RTC_REG_DAYOFWEEK 0x03
#define RTC_REG_DAY       0x04
#define RTC_REG_MONTH     0x05
#define RTC_REG_YEAR      0x06

///
/// Alarm registers.
/// 

#define RTC_REG_A1SECS    0x07
#define RTC_REG_A1MINS    0x08
#define RTC_REG_A1HOURS   0x09
#define RTC_REG_A1DOW     0x0A
#define RTC_REG_A1DAY     0x0A

#define RTC_REG_A2MINS    0x0B
#define RTC_REG_A2HOURS   0x0C
#define RTC_REG_A2DOW     0x0D
#define RTC_REG_A2DAY     0x0D

///
/// Control and status registers.
///

#define RTC_REG_CONTROL   0x0E
#define RTC_REG_STATUS    0x0F

#define RTC_LAST_REG      RTC_REG_STATUS

/// 
/// RTC Control Bits
///

#define RTC_EOSC          0x80
#define RTC_RS2           0x10
#define RTC_RS1           0x08
#define RTC_INTCN         0x04
#define RTC_A2IE          0x02
#define RTC_A1IE          0x01

///
/// status bits
///

#define RTC_OSF           0x80
#define RTC_A2F           0x02
#define RTC_A1F           0x01

///
/// Alarm modes
/// 
/// bits  DY A1M4 A1M3 A1M2 A1M1

#define RTC_ALARM1_MODE1  0x0F ///< once per second
#define RTC_ALARM1_MODE2  0x0E ///< seconds match
#define RTC_ALARM1_MODE3  0x0C ///< minutes and seconds match
#define RTC_ALARM1_MODE4  0x08 ///< hours, minutes, seconds, match
#define RTC_ALARM1_MODE5  0x00 ///< day, hours, minutes, seconds, match
#define RTC_ALARM1_MODE6  0x10 ///< dow, hours, minutes, seconds, match

///
/// AM or PM
///

#define PM 1
#define AM 0

class RTC {
 private:
  // EEPROM Registers
  union {
    struct {
      unsigned i2c_addr   : 8;

      unsigned secs       : 4;
      unsigned secs_10    : 3;
      unsigned UNUSED_0   : 1;
      
      unsigned mins       : 4;
      unsigned mins_10    : 3;
      unsigned UNUSED_1   : 1;

      unsigned hours      : 4;
      unsigned hours_10   : 2; // pm is msb
      unsigned hour12     : 1;
      unsigned UNUSED_2   : 1;

      unsigned dow        : 3;
      unsigned UNUSED_3   : 5;
    
      unsigned day        : 4;
      unsigned day_10     : 2;
      unsigned UNUSED_4   : 2;
      
      unsigned month      : 4;
      unsigned month_10   : 1;
      unsigned century    : 1;
      unsigned UNUSED_5   : 2;
    
      unsigned year       : 4;
      unsigned year_10    : 4;

      unsigned a1_secs       : 4;
      unsigned a1_secs_10    : 3;
      unsigned a1m1          : 1;
      
      unsigned a1_mins       : 4;
      unsigned a1_mins_10    : 3;
      unsigned a1m2          : 1;

      unsigned a1_hours      : 4;
      unsigned a1_hours_10   : 1;
      unsigned a1_pm         : 1;
      unsigned a1_hour12     : 1;
      unsigned a1m3          : 1;

      unsigned a1_day        : 4;
      unsigned a1_day10      : 2;
      unsigned a1_dy         : 1;
      unsigned a1m4          : 1;
    
      unsigned a2_mins       : 4;
      unsigned a2_mins_10    : 3;
      unsigned a2m2          : 1;

      unsigned a2_hours      : 4;
      unsigned a2_hours_10   : 1;
      unsigned a2_pm         : 1;
      unsigned a2_hour12     : 1;
      unsigned a2m3          : 1;

      unsigned a2_day        : 4;
      unsigned a2_day_10     : 2;
      unsigned a2_dy         : 1;
      unsigned a2m4          : 1;

      unsigned a1ie          : 1;
      unsigned a2ie          : 1;
      unsigned intcn         : 1;
      unsigned rs1           : 1;
      unsigned rs2           : 1;
      unsigned UNUSED_6      : 2;
      unsigned n_eosc        : 1;
      
      unsigned a1f           : 1;
      unsigned a2f           : 1;
      unsigned UNUSED_7      : 5;
      unsigned osf           : 1;
    
    } reg_bits;
    struct {
      unsigned char i2c_addr;
      unsigned char secs;
      unsigned char mins;
      unsigned char hours;
      unsigned char dayofweek;
      unsigned char day;
      unsigned char month;
      unsigned char year;

      unsigned char a1_secs;
      unsigned char a1_mins;
      unsigned char a1_hours;
      unsigned char a1_day;

      unsigned char a2_mins;
      unsigned char a2_hours;
      unsigned char a2_day;

      unsigned char control;
      unsigned char status;
    } reg_bytes;
    unsigned char reg_array[16];
  } _regs;
  unsigned char _buf[16];
 public:
  static const unsigned char _i2c_rd_address = (0x68 << 1) | 0x01; ///< I2C read address (8 bit)
  static const unsigned char _i2c_wr_address = (0x68 << 1) | 0x00; ///< I2C write address (8 bit)
  RTC();
  char write_reg_ptr(unsigned char reg);
  char write_reg(unsigned char reg, unsigned char v);
  char write_regs(unsigned char reg, unsigned char *v, unsigned char n);
  unsigned char read_reg(unsigned char reg);
  char read_regs();
  unsigned char get_secs();
  unsigned char get_mins();
  unsigned char get_hours();
  unsigned char get_day();  
  unsigned char get_month();
  unsigned int  get_year();
  unsigned char set_year(unsigned int year);
  unsigned char set_month(unsigned char month);
  unsigned char set_day(unsigned char day);
  unsigned char set_date(unsigned int year, unsigned char month, 
			 unsigned char day);
  unsigned char set_secs(unsigned char secs);
  unsigned char set_mins(unsigned char mins);
  unsigned char set_hours12(unsigned char hours, unsigned char pm_p);
  unsigned char set_hours(unsigned char hours);
  unsigned char set_time(unsigned char hours, unsigned char mins, 
			 unsigned char secs);
  unsigned char set_time12(unsigned char hours, unsigned char mins, 
			   unsigned char secs, unsigned char pm_p);
  unsigned char set_alarm1(unsigned char mode, unsigned char day_or_dow,
			   unsigned char hours, unsigned char mins, 
			   unsigned char secs);
  unsigned char enable_alarm1(void);
  unsigned char clear_alarm1(void);
  void localtime(char *str);
  unsigned char get_reg(unsigned char reg);
};

#endif
